#include <bits/stdc++.h>
using namespace std;

struct Fenwick {
    int n;
    vector<long long> f;
    Fenwick(int n_ = 0) { init(n_); }
    void init(int n_) { n = max(0, n_); f.assign(n + 1, 0); }
    void add(int idx, long long delta) {
        if (idx <= 0) return;
        if (n == 0) return;
        if (idx > n) idx = n; 
        for (int i = idx; i <= n; i += i & -i) f[i] += delta;
    }
    long long sum(int idx) const {
        if (n == 0) return 0;
        if (idx <= 0) return 0;
        if (idx > n) idx = n;
        long long s = 0;
        for (int i = idx; i > 0; i -= i & -i) s += f[i];
        return s;
    }
    long long range_sum(int l, int r) const {
        if (n == 0) return 0;
        if (r < l) return 0;
        if (r < 1) return 0;
        if (l > n) return 0;
        if (l < 1) l = 1;
        if (r > n) r = n;
        return sum(r) - sum(l - 1);
    }
};

struct Solve {
    int n;
    vector<int> a; 
    set<int> ones;
    Fenwick bitCnt, bitSum; 
    int onesCnt;
    int pairs;

    Solve() : n(0), onesCnt(0), pairs(0) {}

    void init_from_string(const string &s) {
        n = (int)s.size();
        a.assign(n + 1, 0);
        for (int i = 1; i <= n; ++i) a[i] = (s[i - 1] == '1') ? 1 : 0;
        bitCnt.init(n);
        bitSum.init(n);
        ones.clear();
        for (int i = 1; i <= n; ++i) if (a[i]) ones.insert(i);
        onesCnt = (int)ones.size();
        pairs = 0;
        for (int i = 2; i <= n; ++i) if (a[i] && a[i - 1]) ++pairs;

        int prev = -1;
        for (int pos : ones) {
            if (prev != -1) {
                int gap = pos - prev - 1;
                if (gap > 0) {
                    bitCnt.add(gap, 1);
                    bitSum.add(gap, gap);
                }
            }
            prev = pos;
        }
    }

    inline void addGap(int len, int delta) {
        if (len <= 0) return;
        if (len > n) len = n;
        bitCnt.add(len, delta);
        bitSum.add(len, (long long)len * delta);
    }

    void flip(int p) {
        if (p < 1 || p > n) return; 
        if (a[p] == 0) {
            // 0 -> 1
            auto it = ones.lower_bound(p);
            int nxt = (it == ones.end() ? -1 : *it);
            int prev = -1;
            if (it != ones.begin()) { auto it2 = it; --it2; prev = *it2; }

            if (prev == p - 1) ++pairs;
            if (nxt == p + 1) ++pairs;

            if (prev != -1 && nxt != -1) {
                int oldg = nxt - prev - 1;
                if (oldg > 0) addGap(oldg, -1);
            }
            if (prev != -1) {
                int lg = p - prev - 1;
                if (lg > 0) addGap(lg, +1);
            }
            if (nxt != -1) {
                int rg = nxt - p - 1;
                if (rg > 0) addGap(rg, +1);
            }
            a[p] = 1;
            ones.insert(p);
            ++onesCnt;
        } else {
            // 1 -> 0
            auto it = ones.find(p);
            int nxt = -1, prev = -1;
            if (it != ones.end()) {
                auto itn = it; ++itn; if (itn != ones.end()) nxt = *itn;
                if (it != ones.begin()) { auto itp = it; --itp; prev = *itp; }
            }
            if (prev == p - 1) --pairs;
            if (nxt == p + 1) --pairs;

            if (prev != -1) {
                int lg = p - prev - 1;
                if (lg > 0) addGap(lg, -1);
            }
            if (nxt != -1) {
                int rg = nxt - p - 1;
                if (rg > 0) addGap(rg, -1);
            }
            if (prev != -1 && nxt != -1) {
                int ng = nxt - prev - 1;
                if (ng > 0) addGap(ng, +1);
            }
            a[p] = 0;
            ones.erase(p);
            --onesCnt;
        }
    }

    long long take_by_budget(int t) {
        if (t <= 0 || n == 0) return 0;
        long long totalSum = bitSum.sum(n);
        if (totalSum <= t) return bitCnt.sum(n);

        int lo = 1, hi = n, idx = 0;
        while (lo <= hi) {
            int mid = (lo + hi) >> 1;
            if (bitSum.sum(mid) <= t) { idx = mid; lo = mid + 1; }
            else hi = mid - 1;
        }
        long long res = bitCnt.sum(idx);
        long long used = bitSum.sum(idx);
        long long rem = t - used;
        int nextLen = idx + 1;
        if (nextLen <= n && rem > 0) {
            long long avail = bitCnt.range_sum(nextLen, nextLen);
            if (avail > 0) res += min(avail, rem / nextLen);
        }
        return res;
    }

    int query(int x) {
        if (n == 0) return 0;
        int zeros = n - onesCnt;
        int t = min(x, zeros);
        if (onesCnt == 0) return max(0, t - 1);
        long long filled = take_by_budget(t);
        return pairs + t + (int)filled;
    }
};

int main() {
    ios::sync_with_stdio(false);
    cin.tie(nullptr);

    int n, q;
    if (!(cin >> n >> q)) return 0;
    string s;
    cin >> s;
    Solve S;
    S.init_from_string(s);

    while (q--) {
        int typ, x;
        cin >> typ >> x;
        if (typ == 1) {
            S.flip(x);
        } else {
            cout << S.query(x) << '\n';
        }
    }
    return 0;
}
